<?php
/**
 * Transfer Ruud
 * https://webenginecms.org/
 * 
 * @version 1.1.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2021 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\TransferRuud;

class TransferRuud {
	
	private $_configXml = 'config.xml';
	private $_modulesPath = 'modules';
	
	private $_enableTransferTax = 0;
	private $_transferTax = 10;
	private $_creditConfig = 0;
	private $_transferMinLimit = 10;
	private $_transferMaxLimit = 10000;
	private $_ruudColumnName = 'Ruud';
	
	private $_user;
	private $_player;
	private $_senderPlayer;
	private $_transferAmount;
	
	private $_usercpmenu = array(
		array(
			'active' => true,
			'type' => 'internal',
			'phrase' => 'transferruud_title',
			'link' => 'ruud/transfer',
			'icon' => 'usercp_default.png',
			'visibility' => 'user',
			'newtab' => false,
			'order' => 999,
		),
	);
	
	// CONSTRUCTOR
	
	function __construct() {
		
		// load databases
		$this->common = new \common();
		
		// config file path
		$this->configFilePath = __PATH_TRANSFERRUUD_ROOT__.$this->_configXml;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('transferruud_error_2', true));
		$xml = simplexml_load_file($this->configFilePath);
		if(!$xml) throw new \Exception(lang('transferruud_error_2', true));
		$this->_configs = convertXML($xml->children());
		if(!is_array($this->_configs)) throw new \Exception(lang('transferruud_error_2', true));
		
		// set configs
		$this->_enableTransferTax = $this->_configs['enable_transfer_tax'];
		$this->_transferTax = $this->_configs['transfer_tax'];
		$this->_creditConfig = $this->_configs['credit_config'];
		$this->_transferMinLimit = $this->_configs['transfer_minimum_limit'];
		$this->_transferMaxLimit = $this->_configs['transfer_maximum_limit'];
		
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('transferruud_error_4', true));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('transferruud_error_4', true));
		if(!@include_once(__PATH_TRANSFERRUUD_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('transferruud_error_4', true));
	}
	
	public function setUser($user) {
		$this->_user = $user;
	}
	
	public function setPlayer($player) {
		$this->_player = $player;
	}
	
	public function setSenderPlayer($player) {
		$this->_senderPlayer = $player;
	}
	
	public function getTransferTaxValue() {
		if($this->_enableTransferTax == 0) return 0;
		return $this->_transferTax;
	}
	
	public function setTransferAmount($amount) {
		if(!\Validator::UnsignedNumber($amount)) throw new \Exception(lang('transferruud_error_7',true));
		if($amount < $this->_transferMinLimit) throw new \Exception(lang('transferruud_error_7',true));
		if($amount > $this->_transferMaxLimit) throw new \Exception(lang('transferruud_error_7',true));
		$this->_transferAmount = $amount;
	}
	
	public function calculateTransferTotal() {
		if(!check_value($this->_transferAmount)) return 0;
		if($this->_transferAmount == 0) return 0;
		
		$tax = $this->getTransferTaxValue();
		if($tax == 0) return $this->_transferAmount;
		
		$result = round($this->_transferAmount+($this->_transferAmount*($tax/100)));
		return $result;
	}
	
	public function transfer() {
		if($this->_creditConfig == 0) throw new \Exception(lang('transferruud_error_15'));
		if(!check_value($this->_user)) throw new \Exception(lang('transferruud_error_14'));
		if(!check_value($this->_player)) throw new \Exception(lang('transferruud_error_5'));
		if(!check_value($this->_senderPlayer)) throw new \Exception(lang('transferruud_error_6'));
		if(!check_value($this->_transferAmount)) throw new \Exception(lang('transferruud_error_7'));
		
		
		// validate sender player
		if(!$this->_validateSenderPlayer()) throw new \Exception(lang('transferruud_error_6'));
		
		// validate sender player ruud amount
		if(!$this->_validateSenderPlayerRuudAmount()) throw new \Exception(lang('transferruud_error_10'));
		
		// check sender player account online status
		if($this->common->accountOnline($this->_user)) throw new \Exception(lang('error_14'));
		
		// transfer total
		$transferTotal = $this->calculateTransferTotal();
		
		// check receiving player
		$Character = new \Character();
		$characterData = $Character->CharacterData($this->_player);
		if(!is_array($characterData)) throw new \Exception(lang('transferruud_error_12'));
		
		// receiver player account
		$receiverAccount = $characterData[_CLMN_CHR_ACCID_];
		
		// receiver player name
		$receiverPlayer = $characterData[_CLMN_CHR_NAME_];
		
		// check receiver player account online status
		if($this->common->accountOnline($receiverAccount)) throw new \Exception(lang('transferruud_error_13'));
		
		try {
			
			// subtract ruud from sender player
			$creditSystem = new \CreditSystem();
			$creditSystem->setConfigId($this->_creditConfig);
			$configInfo = $creditSystem->showConfigs(true);
			if($configInfo['config_user_col_id'] != 'character') throw new \Exception(lang('error_73',true));
			$creditSystem->setIdentifier($this->_senderPlayer);
			$creditSystem->subtractCredits($transferTotal);
			
			// add ruud to receiver player
			$creditSystem->setIdentifier($receiverPlayer);
			$creditSystem->addCredits($this->_transferAmount);
			
		} catch(\Exception $ex) {
			if(config('error_reporting',true)) {
				throw new \Exception($ex->getMessage());
			} else {
				throw new \Exception(lang('transferruud_error_9'));
			}
		}
		
		redirect(1,'ruud/transfer/success/1/');
	}
	
	public function getAccountPlayerListRuud() {
		if(!check_value($this->_user)) return;
		$Character = new \Character();
		$AccountCharacters = $Character->AccountCharacter($this->_user);
		if(!is_array($AccountCharacters)) return;
		foreach($AccountCharacters as $player) {
			$characterData = $Character->CharacterData($player);
			if(!array_key_exists($this->_ruudColumnName, $characterData)) throw new \Exception(lang('transferruud_error_3'));
			$result[$characterData[_CLMN_CHR_NAME_]] = $characterData[$this->_ruudColumnName];
		}
		if(!is_array($result)) return;
		return $result;
	}
	
	public function checkPluginUsercpLinks() {
		if(!is_array($this->_usercpmenu)) return;
		$cfg = loadConfig('usercp');
		if(!is_array($cfg)) return;
		foreach($cfg as $usercpMenu) {
			$usercpLinks[] = $usercpMenu['link'];
		}
		foreach($this->_usercpmenu as $pluginMenuLink) {
			if(in_array($pluginMenuLink['link'],$usercpLinks)) continue;
			$cfg[] = $pluginMenuLink;
		}
		usort($cfg, function($a, $b) {
			return $a['order'] - $b['order'];
		});
		$usercpJson = json_encode($cfg, JSON_PRETTY_PRINT);
		$cfgFile = fopen(__PATH_CONFIGS__.'usercp.json', 'w');
		if(!$cfgFile) throw new \Exception('There was a problem opening the usercp file.');
		fwrite($cfgFile, $usercpJson);
		fclose($cfgFile);
	}
	
	// PRIVATE FUNCTIONS
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_TRANSFERRUUD_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _validateSenderPlayer() {
		if(!check_value($this->_senderPlayer)) return;
		$accountPlayerList = $this->getAccountPlayerListRuud();
		if(!is_array($accountPlayerList)) return;
		if(!array_key_exists($this->_senderPlayer, $accountPlayerList)) return;
		return true;
	}
	
	private function _validateSenderPlayerRuudAmount() {
		if(!check_value($this->_senderPlayer)) return;
		if(!check_value($this->_transferAmount)) return;
		
		$accountPlayerList = $this->getAccountPlayerListRuud();
		if(!is_array($accountPlayerList)) return;
		if(!array_key_exists($this->_senderPlayer, $accountPlayerList)) return;
		
		$transferTotal = $this->calculateTransferTotal();
		if($transferTotal == 0) return;
		
		if($transferTotal > $accountPlayerList[$this->_senderPlayer]) return;
		return true;
	}
	
}